# Output Format & Logo Management Guide

## Question 1: PNG vs MP4 Output

### Current Status:
**Most scripts output PNGs**, but 5 scripts output MP4s:

**MP4 Output (H.264):**
- `run_TdAnimation3.py`
- `run_FgAnimation3.py`
- `run_2ptAnimation3.py`
- `run_Rotate3Dtext-2PT.py`
- `run_Rotate3Dtext-FG.py`

**PNG Sequence Output:**
- All other 23+ scripts

### To Change MP4 to PNG:

Edit the Python runner file and change:
```python
# FROM:
OM_TEMPLATE = "H.264 - Match Source - High Bitrate"
EXT = ".mp4"

# TO:
OM_TEMPLATE = "PNG Sequence"
EXT = ""  # Empty for PNG sequences
```

### To Change PNG to MP4:

Edit the Python runner file and change:
```python
# FROM:
OM_TEMPLATE = "PNG Sequence"
EXT = ""

# TO:
OM_TEMPLATE = "H.264 - Match Source - High Bitrate"
EXT = ".mp4"
```

---

## Question 2: Changing Logo Directory

### Where to Change:

The logo directory is defined in **each Python runner** at the top of the file:

```python
LOGO_DIR = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Large"
```

### To Use a Different Size:

**Option A: Change Individual Scripts**
Edit each Python runner file:
```python
# For small logos:
LOGO_DIR = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Small"

# For medium logos:
LOGO_DIR = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Medium"
```

**Option B: Create Size Variants in Same Directory**
Organize logos like this:
```
Logos/
  ├── Small/
  │   ├── NFL/
  │   └── MLB/
  ├── Medium/
  │   ├── NFL/
  │   └── MLB/
  └── Large/
      ├── NFL/
      └── MLB/
```

Then change:
```python
LOGO_DIR = "/path/to/Logos/Small"  # or Medium, or Large
```

### Global Change (All Scripts):

To change all scripts at once:
```bash
cd "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations"

# Change from Logos-Large to Logos-Medium
sed -i.bak 's|Logos-Large|Logos-Medium|g' run_*.py

# Or to a completely different path:
sed -i.bak 's|/Users/.*/Logos-Large|/path/to/new/logos|g' run_*.py
```

---

## Question 3: Clear Logo Assets After Render

This is a great idea to keep the AE project clean! We need to add logo cleanup to `gl_common.jsxinc`.

### Solution: Add Cleanup Function to gl_common.jsxinc

Add this function to `gl_common.jsxinc`:

```javascript
// --- Logo cleanup (remove imported logos to keep project clean) ---
GL.cleanupImportedLogos = function(){
  var removed = 0;
  var logoPattern = /\.(png|jpg|jpeg|svg|ai|psd)$/i;

  for (var i = app.project.numItems; i >= 1; i--){
    var item = app.project.item(i);
    try{
      if (item instanceof FootageItem && item.file){
        var filename = item.file.name.toLowerCase();
        var parentFolder = item.file.parent ? item.file.parent.name.toLowerCase() : "";

        // Check if this looks like a logo file
        // Match: files from "logos", "NFL", "MLB", etc. folders
        if (logoPattern.test(filename) &&
            (parentFolder.indexOf("logo") !== -1 ||
             parentFolder === "nfl" || parentFolder === "mlb" ||
             parentFolder === "nba" || parentFolder === "nhl" ||
             parentFolder === "cfb" || parentFolder.length === 3)){

          // Only remove if no comp is using it (safety check)
          if (item.usedIn && item.usedIn.length === 0){
            item.remove();
            removed++;
          }
        }
      }
    }catch(e){}
  }
  return removed;
};
```

### Then Update JSX Scripts to Call Cleanup:

At the **end** of the render loop in your JSX scripts, add:

```javascript
// After all rendering is complete
app.endUndoGroup();

// Clean up imported logos
var cleaned = GL.cleanupImportedLogos();
$.writeln("Cleaned up " + cleaned + " logo assets from project");

if (app.endSuppressDialogs){ try{ app.endSuppressDialogs(); }catch(e){} }
if (QUIT_APP) app.quit();
```

### Alternative: Clean Only Unused Assets

A safer approach that removes ALL unused footage (not just logos):

```javascript
GL.cleanupUnusedFootage = function(){
  var removed = 0;
  for (var i = app.project.numItems; i >= 1; i--){
    var item = app.project.item(i);
    try{
      if (item instanceof FootageItem){
        // Only remove if nothing is using it
        if (!item.usedIn || item.usedIn.length === 0){
          item.remove();
          removed++;
        }
      }
    }catch(e){}
  }
  return removed;
};
```

---

## Implementation Steps

### Step 1: Update gl_common.jsxinc

Add one of the cleanup functions above to the end of `gl_common.jsxinc` (before the final `$.global.GL = GL;`).

### Step 2: Update JSX Scripts

For scripts that already use GL library, add the cleanup call at the very end:

```javascript
// Example in batch_ScoreAnimations1-TD.jsx
app.endUndoGroup();

// NEW: Clean up logos
if (typeof GL.cleanupImportedLogos === "function"){
  var cleaned = GL.cleanupImportedLogos();
  $.writeln("Removed " + cleaned + " logo assets from project");
}

if (app.endSuppressDialogs){ try{ app.endSuppressDialogs(); }catch(e){} }
if (QUIT_APP) app.quit();
```

### Step 3: Enable/Disable Cleanup

You can control cleanup via environment variable:

**In Python runner:**
```python
env = {
    # ... other settings ...
    "AE_CLEANUP_LOGOS": "1",  # Set to "0" to disable cleanup
}
```

**In JSX:**
```javascript
var CLEANUP_LOGOS = (GL.env("AE_CLEANUP_LOGOS","1") === "1");

// At the end:
if (CLEANUP_LOGOS && typeof GL.cleanupImportedLogos === "function"){
  var cleaned = GL.cleanupImportedLogos();
  $.writeln("Removed " + cleaned + " logo assets");
}
```

---

## Safety Notes

1. **Cleanup only removes UNUSED footage** - if a comp references a logo, it won't be removed
2. **Run cleanup AFTER rendering** - don't clean up before all renders are complete
3. **Test with AE_NO_RENDER="1" first** - verify cleanup works without rendering
4. **Logos are NOT deleted from disk** - only removed from the AE project file

---

## Summary

**Q1: PNG vs MP4?**
- Most scripts output PNG sequences
- 5 scripts (Animation 3 variants + Rotate3D) output MP4
- Change `OM_TEMPLATE` and `EXT` in Python runner to switch

**Q2: Change logo size?**
- Edit `LOGO_DIR` in each Python runner file
- Or use sed command to batch-change all scripts

**Q3: Clean up logos?**
- Add `GL.cleanupImportedLogos()` function to gl_common.jsxinc
- Call it at the end of JSX scripts after rendering
- Only removes unused footage items (safety feature)

---

**Would you like me to implement the logo cleanup function now?**
